//==================================================================================================
// Author Jens Chluba Aug/Sept 2010
// purpose: explicit expression for matrix elements of the first few ns & nd-states in HI
// last modification: Aug 2012
//==================================================================================================
// 18th Aug, 2012: checked matrix elements with Mathematica

#include <cmath>

#include "Definitions.h"
#include "HI_matrix_elements.h"

using namespace std;

//==================================================================================================
//==================================================================================================
namespace HI_matrix_elements 
{
    double Rnsnp(int n){ return -1.5*n*sqrt(n*n - 1.0); }
    double Rndnp(int n){ return -1.5*n*sqrt(n*n - 4.0); }
    
    //==============================================================================================
    // bound-bound matrix elements
    //==============================================================================================
    double R1snp(int n){ double dn=n; return 16.0 *pow((dn-1.0)/(dn+1.0), n)*sqrt( pow(dn, 7)
                                                  /pow(dn*n-1.0, 5) ); }
    
    double R2snp(int n){ double dn=n; return 256.0*sqrt( 2.0*pow(dn, 7)*(dn*n-1.0) )
                                                  *pow((dn-2.0)/(n+2.0), n)/pow(dn*n-4.0, 3); }
    
    //==============================================================================================
    double R3snp(int n){ double dn=n; return 432.0*sqrt( 3.0*pow(dn, 7)*(dn*n-1.0) )*(7.0*n*n-27.0)
                                                  *pow((dn-3.0)/(n+3.0), n)/pow(dn*n-9.0, 4); }
    
    double R3dnp(int n){ double dn=n; return 864.0*sqrt( 1.2*pow(dn,11)*(dn*n-1.0) )               
                                                  *pow((dn-3.0)/(dn+3.0), n)/pow(dn*n-9.0, 4); }
    
    //==============================================================================================
    double R4snp(int n){ double dn=n; return 64.0/3.0*sqrt( 1.0*pow(4.0*n, 7)*(dn*n-1.0) )
                                                     *((23.0*dn*n-288.0)*dn*n+768.0)
                                                     *pow((dn-4.0)/(dn+4.0), n)
                                                     /pow(dn*n-16.0, 5); }
    
    double R4dnp(int n){ double dn=n; return 8.0 /3.0*sqrt( 0.2*pow(4.0*n,11)*(dn*n-1.0) )
                                                     *(7.0*n*n-48.0)*pow((dn-4.0)/(dn+4.0), n)
                                                     /pow(dn*n-16.0, 5); }
    
    //==============================================================================================
    double R5snp(int n){ double dn=n; return 16.0*625.0/3.0*sqrt( 5.0*pow(dn, 7)*(dn*n-1.0) )
                                                 *(((91.0*n*n-2545.0)*n*n+20625.0)*n*n-46875.0)
                                                 *pow((dn-5.0)/(n+5.0), n)/pow(dn*n-25.0, 6); }
    
    double R5dnp(int n){ double dn=n; return 32.0*625.0/3.0*sqrt( 10.0/7.0*pow(dn,11)*(dn*n-1.0) )
                                                 *((29.0*dn*n-590.0)*dn*n+2625.0)                
                                                 *pow((dn-5.0)/(dn+5.0), n)/pow(dn*n-25.0, 6); }
    
    //==============================================================================================
    double R6snp(int n){ double dn=n; return (6912.0*sqrt(6.0)*pow(dn,3)
                                                    *pow((-6.0 + dn)/(6.0 + dn), n)
                                                    *sqrt(dn*(-1.0 + dn*dn))*(-108.0 + 7.0*dn*dn)
                                                    *(-233280.0 + (97200.0 - (10620.0
                                                      - 289.0*dn*dn)*dn*dn)*dn*dn))
                                                    /(5.0*pow(-36.0 + dn*dn,7));}
    
    double R6dnp(int n){ double dn=n; return (55296.0*sqrt(0.08571428571428572)*pow(dn,5)
                                              *pow((-6.0 + dn)/(6.0 + dn),n)*sqrt(dn*(-1.0 + dn*dn))
                                              *(-326592.0 + (89424.0 - (7092.0 
                                                - 167.0*dn*dn)*dn*dn)*dn*dn))
                                              /pow(-36.0 + dn*dn,7); }
    
    //==============================================================================================
    double R7snp(int n){ double dn=n; return (38416.0*sqrt(7.0)*pow(dn,3)
                                               *pow((-7.0 + dn)/(7.0 + dn), n)
                                               *sqrt(dn*(-1.0 + dn*dn))
                                               *(-12711386205.0 + (6485401125.0 
                                                 -(1097665170.0 - (79704282.0 
                                                 -(2547265.0 - 29233.0*dn*dn)
                                               *dn*dn)*dn*dn)*dn*dn)*dn*dn))
                                               /(45.0*pow(-49.0 + dn*dn,8)); }
    
    double R7dnp(int n){ double dn=n; return (153664.0*sqrt(0.4666666666666667)*pow(dn,5)
                                              *pow((-7.0 + dn)/(7.0 + dn),n)
                                              *sqrt(dn*(-1.0 + dn*dn))
                                              *(155649627.0 -(47799108.0-(4760154.0 
                                                -(186788.0 - 2483.0*dn*dn)
                                              *dn*dn)*dn*dn)*dn*dn))
                                              /(9.0*pow(-49.0 + dn*dn,8)); }
    
    //==============================================================================================
    double R8snp(int n){ double dn=n; return (131072.0*sqrt(2.0)*pow(dn,3)
                                              *pow((-8.0 + dn)/(8.0 + dn),n)
                                              *sqrt(dn*(-1.0 + dn*dn))
                                              *(21646635171840.0 -(11499774935040.0 
                                                -(2101597962240.0 -(175037743104.0 
                                                -(7190401024.0-(140890496.0 
                                                -1044871.0*dn*dn)
                                                 *dn*dn)*dn*dn)*dn*dn)*dn*dn)*dn*dn))
                                              /(315.0*pow(-64.0 + dn*dn,9));}
    
    double R8dnp(int n){ double dn=n; return (262144.0*sqrt(0.01904761904761905)*pow(dn,5)
                                              *pow((-8.0 + dn)/(8.0 + dn),n)
                                              *sqrt(dn*(-1.0 + dn*dn))
                                              *(-1014686023680.0 +(335963750400.0 
                                                -(38338560000.0 -(1932468224.0 
                                                -(43842880.0 -363461.0*dn*dn)
                                                 *dn*dn)*dn*dn)*dn*dn)*dn*dn))
                                              /(45.0*pow(-64.0 + dn*dn,9)); }
        
    //==============================================================================================
    double Rksnp(int k, int n)
    {
        if(k==n) return Rnsnp(n);
        if(k==1) return R1snp(n);
        if(k==2) return R2snp(n);
        if(k==3) return R3snp(n);
        if(k==4) return R4snp(n);
        if(k==5) return R5snp(n);
        if(k==6) return R6snp(n);
        if(k==7) return R7snp(n);
        if(k==8) return R8snp(n);
        
        return 0.0;
    }

    double Rkdnp(int k, int n)
    {
        if(k==n) return Rndnp(n);
        if(k==3) return R3dnp(n);
        if(k==4) return R4dnp(n);
        if(k==5) return R5dnp(n);
        if(k==6) return R6dnp(n);
        if(k==7) return R7dnp(n);
        if(k==8) return R8dnp(n);
        
        return 0.0;
    }
    
    //==============================================================================================
    // bound-free matrix elements
    //==============================================================================================
    double C1s(double x){ return 16.0 *exp( -2.0*atan(x)/x )*sqrt( x/pow(1.0+x*x, 5)
                                       /( 1.0-exp(-2.0*PI/x) )); }
    
    double C2s(double x){ return 256.0*exp( -2.0*atan(2.0*x)/x )*sqrt( 2.0*x*(1.0+x*x)
                                       /( 1.0-exp(-2.0*PI/x) ) ) /pow(1.0+4.0*x*x, 3); }
    
    //==============================================================================================
    double C3s(double x){ return 432.0*exp( -2.0*atan(3.0*x)/x )*(7.0+27.0*x*x)
                                      *sqrt( 3.0*x*(1.0+x*x)/( 1.0-exp(-2.0*PI/x) ) ) 
                                      /pow(1.0+9.0*x*x, 4); }
    
    double C3d(double x){ return 864.0*exp( -2.0*atan(3.0*x)/x )               
                                      *sqrt( 1.2*x*(1.0+x*x)/( 1.0-exp(-2.0*PI/x) ) ) 
                                      /pow(1.0+9.0*x*x, 4); }
    
    //==============================================================================================
    double C4s(double x){ return 8192.0/3.0*exp( -2.0*atan(4.0*x)/x )
                                           *(23.0+96.0*x*x*(3.0+8.0*x*x) )
                                           *sqrt(    x*(1.0+x*x)/( 1.0-exp(-2.0*PI/x) ) ) 
                                           /pow(1.0+16.0*x*x, 5); }
    
    double C4d(double x){ return 8192.0/3.0*exp( -2.0*atan(4.0*x)/x )*(7.0+48.0*x*x)                
                                           *sqrt(0.8*x*(1.0+x*x)/( 1.0 - exp(-2.0*PI/x) ) ) 
                                           /pow(1.0+16.0*x*x, 5); }
    
    //==============================================================================================
    double C5s(double x){ return 1.0e+4/3.0*exp( -2.0*atan(5.0*x)/x )
                                           *(91.0+5.0*x*x*(509.0+x*x*(4125.0+9375.0*x*x)) )
                                           *sqrt(5.0*x*(1.0+x*x)/( 1.0 - exp(-2.0*PI/x) ) ) 
                                           /pow(1.0+25.0*x*x, 6); }
    
    double C5d(double x){ return 2.0e+4/3.0*exp( -2.0*atan(5.0*x)/x )
                                           *(29.0+x*x*(590.0+2625.0*x*x) )            
                                           *sqrt(10.0/7.0*x*(1.0+x*x)/( 1.0-exp(-2.0*PI/x) ) ) 
                                           /pow(1.0+25.0*x*x, 6); }
    //==============================================================================================
    double C6s(double x){ return exp( -2.0*atan(6.0*x)/x )/sqrt(1.0 - exp(-2.0*PI/x))*
                                    (6912.0*sqrt( 6.0*x*(1.0 + x*x) )
                                   *(7.0 + 108.0*x*x)*(289.0 + 180.0*x*x
                                   *(59.0 + (540.0 + 1296.0*x*x)*x*x)))
                                    /(5.0*pow(1.0 + 36.0*x*x,7)); }
    
    double C6d(double x){ return exp( -2.0*atan(6.0*x)/x )/sqrt(1.0 - exp(-2.0*PI/x))*
                                    (55296.0*sqrt( 0.08571428571428572*x*(1.0 + x*x) )
                                   *(167.0 + 36.0*x*x
                                   *(197.0 + (2484.0 + 9072.0*x*x)*x*x)))
                                    /pow(1.0 + 36.0*x*x,7); }
    //==============================================================================================
    double C7s(double x){ return exp( -2.0*atan(7.0*x)/x )/sqrt(1.0 - exp(-2.0*PI/x))*
                                    (38416.0*sqrt(7.0*x*(1.0 + x*x))*
                                    (29233.0 + 49.0*x*x*(51985.0 + 21.0*x*x
                                   *(77458.0 + 1715.0*x*x
                                   *(622.0 + (3675.0 + 7203.0*x*x)*x*x)))))/
                                    (45.0*pow(1.0 + 49.0*x*x,8)); }
    
    double C7d(double x){ return exp( -2.0*atan(7.0*x)/x )/sqrt(1.0 - exp(-2.0*PI/x))*
                                    (153664.0*sqrt(0.4666666666666667*x*(1.0 + x*x))*
                                    (2483.0 + 49.0*x*x*(3812.0 + 63.0*x*x
                                   *(1542.0 + (15484.0 + 50421.0*x*x)*x*x))))/
                                    (9.0*pow(1.0 + 49*x*x,8)); }
    //==============================================================================================
    double C8s(double x){ return exp( -2.0*atan(8.0*x)/x )/sqrt(1.0 - exp(-2.0*PI/x))*
                                    (131072*sqrt(2.0*x*(1.0 + x*x))*
                                    (1044871.0 + 128.0*x*x*(1100707.0 + 32.0*x*x
                                   *(1755469.0 + 5376.0*x*x*(7949.0 + 80.0*x*x
                                   *(1193.0 + 384.0*x*x*(17.0 + 32.0*x*x)))))))/
                                    (315.0*pow(1.0 + 64.0*x*x,9)); }
    
    double C8d(double x){ return exp( -2.0*atan(8.0*x)/x )/sqrt(1.0 - exp(-2.0*PI/x))*
                                    (262144.0*sqrt(0.01904761904761905*x*(1.0 + x*x))*
                                    (363461.0 + 64.0*x*x*(685045.0 + 128.0*x*x
                                   *(235897.0 + 2880.0*x*x
                                   *(1625.0 + (14240.0 + 43008.0*x*x)*x*x)))))/
                                    (45.0*pow(1.0 + 64.0*x*x,9)); }

    //==============================================================================================
    double Cks(int k, double x)
    {
        if(k==1) return C1s(x);
        if(k==2) return C2s(x);
        if(k==3) return C3s(x);
        if(k==4) return C4s(x);
        if(k==5) return C5s(x);
        if(k==6) return C6s(x);
        if(k==7) return C7s(x);
        if(k==8) return C8s(x);
        
        return 0.0;
    }
    
    double Ckd(int k, double x)
    {
        if(k==3) return C3d(x);
        if(k==4) return C4d(x);
        if(k==5) return C5d(x);
        if(k==6) return C6d(x);
        if(k==7) return C7d(x);
        if(k==8) return C8d(x);
        
        return 0.0;
    }
}

//==================================================================================================
//==================================================================================================
